/* 
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.log4j.appender;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.GZIPOutputStream;

import org.apache.log4j.helpers.FileHelper;
import org.apache.log4j.helpers.LogLog;

/**
 * @author <a href="mailto:simon_park_mail AT yahoo DOT co DOT uk">Simon Park</a>
 * @version 1.1
 */
final class GzipFileCompressionStrategy extends LogFileCompressionStrategy {

  final String getAlgorithmName() {
    return "gz";
  }

  final boolean isCompressed(final File backupFile) {
    return new FileHelper().isGZip(backupFile);
  }

  final boolean compress(final File backupFile, final File deflatedFile,
      final AppenderRollingProperties properties) {
    InputStream inputStream = null;
    GZIPOutputStream outputStream = null;
    try {
      inputStream = this.createInputStream(backupFile);
      if (inputStream == null) {
        return false;
      }
      outputStream = this.createOutputStream(deflatedFile, properties);
      if (outputStream == null) {
        return false;
      }
      if (!this.compress(inputStream, outputStream)) {
        return false;
      }
    } finally {
      this.closeStreams(inputStream, outputStream);
    }
    return true;
  }

  private InputStream createInputStream(final File file) {
    try {
      return new BufferedInputStream(new FileInputStream(file));
    } catch (FileNotFoundException e) {
      // may occur if backup has been scavenged
      LogLog.debug("Unable to open backup log file", e);
      return null;
    }
  }

  private GZIPOutputStream createOutputStream(final File deflatedFile,
      final AppenderRollingProperties properties) {
    try {
      return new GZIPOutputStream(new BufferedOutputStream(
          new FileOutputStream(deflatedFile)));
    } catch (IOException e) {
      LogLog.warn("Unable to open compressed log file", e);
      return null;
    }
  }

  private void closeStreams(final InputStream inputStream,
      final OutputStream outputStream) {
    if (inputStream != null) {
      try {
        inputStream.close();
      } catch (IOException e) {
        LogLog.warn("Unable to close backup log file", e);
      }
    }
    if (outputStream != null) {
      try {
        outputStream.close();
      } catch (IOException e) {
        LogLog.warn("Unable to close compressed log file", e);
      }
    }
  }

  private boolean compress(final InputStream inputStream,
      final GZIPOutputStream outputStream) {
    try {
      while (inputStream.available() > 0) {
        int data = inputStream.read();
        if (data == -1) {
          break;
        }
        outputStream.write(data);
      }
      outputStream.finish();
      return true;
    } catch (IOException e) {
      LogLog.warn("Unable to transfer data from backup log file"
          + " to compressed log file", e);
      return false;
    }
  }
}